<?php

header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Manejar solicitudes OPTIONS para CORS preflight
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Incluir la conexión a la base de datos
include_once '../conection/db.php';

// Obtener el método de la solicitud
$method = $_SERVER['REQUEST_METHOD'];

try {
    // Recibir el array de banners desde POST (JSON)
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    $banners = isset($data['banners']) ? $data['banners'] : [];
    
    if (empty($banners) || !is_array($banners)) {
        http_response_code(400);
        echo json_encode(["status" => "error", "message" => "No se recibió un array válido de banners"]);
        exit;
    }

    // Obtener todos los IDs actuales en la base de datos
    $stmt = $pdo->query("SELECT id, path_imagen FROM media_secciones");
    $existingBanners = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $existingIds = array_column($existingBanners, 'id');
    
    // Array para almacenar los IDs que vienen en la solicitud
    $receivedIds = [];
    $results = [];

    // Procesar cada banner del array
    foreach ($banners as $index => $banner) {
        $bannerId = isset($banner['id']) ? $banner['id'] : 'new';
        $path_imagen = isset($banner['path_imagen']) ? $banner['path_imagen'] : '';
        $orden = isset($banner['orden']) ? $banner['orden'] : 0;
        $seccion_id = isset($banner['seccion_id']) ? $banner['seccion_id'] : null;
        
        // CASO 1: Crear nuevo banner (id = "new" o no existe)
        if ($bannerId === 'new' || !is_numeric($bannerId)) {
            $query = $pdo->prepare("INSERT INTO media_secciones (path_imagen, orden, seccion_id) VALUES (:path_imagen, :orden, :seccion_id)");
            $query->bindParam(":path_imagen", $path_imagen);
            $query->bindParam(":orden", $orden);
            $query->bindParam(":seccion_id", $seccion_id);
            
            if ($query->execute()) {
                $newId = $pdo->lastInsertId();
                $receivedIds[] = $newId;
                $results[] = [
                    "status" => "success",
                    "action" => "created",
                    "index" => $index,
                    "id" => $newId,
                    "path_imagen" => $path_imagen
                ];
            } else {
                $results[] = [
                    "status" => "error",
                    "index" => $index,
                    "message" => "Error al guardar el banner en la base de datos"
                ];
            }
        }
        // CASO 2: Actualizar banner existente (id numérico)
        else {
            $receivedIds[] = $bannerId;
            
            // Verificar que el banner existe
            $stmtCurrent = $pdo->prepare("SELECT id FROM media_secciones WHERE id = :id");
            $stmtCurrent->bindParam(":id", $bannerId);
            $stmtCurrent->execute();
            $currentBanner = $stmtCurrent->fetch(PDO::FETCH_ASSOC);
            
            if (!$currentBanner) {
                $results[] = [
                    "status" => "error",
                    "index" => $index,
                    "message" => "Banner con ID {$bannerId} no existe"
                ];
                continue;
            }
            
            // Actualizar el registro en la base de datos
            $query = $pdo->prepare("UPDATE media_secciones SET path_imagen = :path_imagen, orden = :orden, seccion_id = :seccion_id WHERE id = :id");
            $query->bindParam(":id", $bannerId);
            $query->bindParam(":path_imagen", $path_imagen);
            $query->bindParam(":orden", $orden);
            $query->bindParam(":seccion_id", $seccion_id);
            
            if ($query->execute()) {
                $results[] = [
                    "status" => "success",
                    "action" => "updated",
                    "index" => $index,
                    "id" => $bannerId,
                    "path_imagen" => $path_imagen
                ];
            } else {
                $results[] = [
                    "status" => "error",
                    "index" => $index,
                    "message" => "Error al actualizar el banner"
                ];
            }
        }
    }
    
    // CASO 3: Eliminar banners que no vienen en el array
    $bannersToDelete = array_diff($existingIds, $receivedIds);
    
    foreach ($bannersToDelete as $deleteId) {
        // Eliminar el registro de la base de datos
        $stmtDelete = $pdo->prepare("DELETE FROM media_secciones WHERE id = :id");
        $stmtDelete->bindParam(":id", $deleteId);
        
        if ($stmtDelete->execute()) {
            $results[] = [
                "status" => "success",
                "action" => "deleted",
                "id" => $deleteId
            ];
        } else {
            $results[] = [
                "status" => "error",
                "action" => "delete_failed",
                "id" => $deleteId,
                "message" => "Error al eliminar el banner"
            ];
        }
    }
    
    // Respuesta final
    echo json_encode([
        "status" => "success",
        "message" => "Operación completada",
        "results" => $results
    ]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        "status" => "error",
        "message" => $e->getMessage()
    ]);
}
